/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/



package cnrg.itx.datax;

import java.util.Vector;
import java.util.Enumeration;
import java.io.Serializable;

/**
 * Class to implement a collection of properties.
 */
public class PropertiesCollection implements Serializable
{
	/**
	 * Type of the properties collection.
	 */
	private int channelType;

	/**
	 * The vector to store all the properties.
	 */
	private Vector properties;

	/**
	 * Default constructor for the PropertiesCollection object.
	 */
	public PropertiesCollection ()
	{
		// Make an empty vector
		properties = new Vector();
	}

	/**
	 * Constructor for the PropertiesCollection object with a specific type.
	 * @param channelType the type of the properties collection
	 */
	public PropertiesCollection (int channelType)
	{
		// Call the default constructor and then initialize the type
		this ();
		this.channelType = channelType;
	}

	/**
	 * Method to set the type of the properties collection object.
	 * @param channelType the type of the properties collection object
	 */
	public void setType (int channelType)
	{
		this.channelType = channelType;
	}

	/**
	 * Method to get the type of the properties collection object.
	 * @return int the type of the properties collection object
	 */
	public int getType ()
	{
		return channelType;
	}
	
	/**
	 * Method to add a property to the collection.
	 * @param p the property to add
	 */
	public void addProperty (Property p)
	{
		properties.addElement (p);
	}

	/**
	 * Method to merge two property collection objects.
	 * @param pc the property collection object to merge
	 * @return PropertiesCollection the merged property collection
	 * @exception DataException thrown if any of the collections are invalid
	 */
	public PropertiesCollection merge(PropertiesCollection  pc) throws DataException
	{
		for (Enumeration e=properties.elements (); e.hasMoreElements (); ) {
			((Property) e.nextElement ()).merge (pc);
		}
		
		return pc;
	}

	/** Method to find the optimal property.
	 * @param pc the property collection to search
	 * @return the optimal property collection
	 */
	public PropertiesCollection findOptimal(PropertiesCollection pc) throws DataException
	{
		for (Enumeration e = properties.elements(); e.hasMoreElements() ; )
		{
			((Property)e.nextElement()).merge(pc);
		}
		
		return pc;
	}

	/**
	 * Method to get the enumerated properties.
	 * @return Enumeration the enumerated properties
	 */
	public Enumeration getProperties()
	{
		return properties.elements();
	}
	
	/**
	 * Tranforms this class into <code>String</code> format.
	 * 
	 * @return the <code>String</code> depiction of <code>PropertiesCollection</code>
	 */
	public String toString()
	{
		String s = new String();
		
		for (Enumeration e = properties.elements(); e.hasMoreElements(); )
		{
			s = s + e.nextElement().toString();
		}
		
		return s;
	}
}
